#include "stdafx.h"
#include "version.h"
#include "RwExp.h"

#include "max.h"
#include "BspExp.h"
#include "common.h"
#include "RwCommon.h"
#include "BspDlg.h"
#include "utilities.h"
#include "rpcollis.h"
#include "osutils.h"
#include "warningsdlg.h"

//extension of report file generated after export (same name as BSP)
#define ReportFileExtension ".TXT"

static Interface *BSPExportIP;

BSPExport::BSPExport(void)
{
    m_importWorld = NULL;        
    m_nodeMaterial = NULL;
    m_nodeLTM = NULL;
}

BSPExport::~BSPExport(void)
{
}

int
BSPExport::ExtCount(void)
{
    return (1);
}

const TCHAR *
BSPExport::Ext(int n)
{
    const TCHAR * result = _T("");
    
    switch (n)
    {
        case 0:
        {
            result = _T("BSP");

            break;
        }
    }

    return result;
}

const TCHAR *
BSPExport::LongDesc(void)
{
    return _T(BSPLongDescription);
}

const TCHAR *
BSPExport::ShortDesc(void)
{
    return _T(BSPShortDescription);
}

const TCHAR *
BSPExport::AuthorName(void)
{            
    return _T(AUTHORNAME);
}

const TCHAR *
BSPExport::CopyrightMessage(void)
{    
    return _T(COPYRIGHT);
}

const TCHAR *
BSPExport::OtherMessage1(void) 
{        
    return _T("");
}

const TCHAR *
BSPExport::OtherMessage2(void)
{        
    return _T("");
}

unsigned int
BSPExport::Version(void)
{                
    return (VERSION);
}

void
BSPExport::ShowAbout(HWND hWnd)
{            
}

int
BSPExport::DoExport(const TCHAR *filename, ExpInterface *ei, Interface *gi, BOOL suppressPrompts, DWORD options) 
{
    CBspDlg dlg;
    rootNode = gi->GetRootNode();
    BOOL status = FALSE;
    
    /* Get a copy of the interface */
    BSPExportIP = gi;
    hGMaxHWnd = gi->GetMAXHWnd();

    /* setup & display the BSP export dialog */
    /* suppress prompts is included for the batch export from maxscript.
       This option simply exports using the current registry settings,
       which have hopefully been loaded in from a reg file. */
    dlg.m_suppressPrompts=suppressPrompts;
    if (dlg.DoModal() == IDOK)
    {
        m_nScaleFactor = dlg.m_nScaleFactor;        
        m_nWorldSectorSize = dlg.m_nWorldSectorSize;        
        m_nWorldSectorMaxPolys = dlg.m_nWorldSectorMaxPolys;
        m_maxPlaneCheck = dlg.m_maxPlaneCheck;
        m_weldThreshold = dlg.m_weldThreshold;
        m_triStripMesh = dlg.m_triStripMesh;
        m_noAlphaInOverlap = dlg.m_noAlphaInOverlap;
        m_maxSectorOverlap = dlg.m_maxSectorOverlap;
        m_limitUVs = dlg.m_limitUVs;
        m_limitUVMax = dlg.m_limitUVMax;
        if (m_limitUVMax > 1)
        {
            /*  Since we only limit low uv coord of each polyogn
                subtracting 1 will ensure all triangles with 1 
                uv tile or less will be mapped 0->m_limitUVMax */
               m_limitUVMax--;
        }        
        m_weldThresholdUV = dlg.m_weldThresholdUV;
        m_backfaceNormals = dlg.m_backfaceNormals;
        m_2SidedMaterials = dlg.m_2SidedMaterials;
        m_colorVertexPrelight = dlg.m_colorVertexPrelight;
        m_weldVertices = dlg.m_weldVertices;
        m_texturenameCase = dlg.m_texturenameCase;
        m_conditionGeometry = dlg.m_conditionGeometry;
        m_exportNormals = dlg.m_exportNormals;
        m_globalWorldOffset.x = dlg.m_worldOffsetX;
        m_globalWorldOffset.y = dlg.m_worldOffsetY;
        m_globalWorldOffset.z = dlg.m_worldOffsetZ;
        m_generateReportFile = dlg.m_generateReportFile;
        m_displayReport = dlg.m_displayReport;
    }
    else
    {
        return (TRUE);
    }
    

    /* open RenderWare library */
    m_rwlib = new RwLib(hGMaxHWnd, FALSE, FALSE, FALSE, FALSE, TRUE, FALSE, FALSE);
    
    /* set up a wait cursor */
    CWaitCursor waitCursor;

    if (!rootNode)
    {
        MessageBox(hGMaxHWnd, "No Root Node", BSPExporterTitle, MB_OK );
        delete(m_rwlib);    
        return (TRUE);
    }

    /* check RW is setup */
    if (!m_rwlib->Opened())
    {
        MessageBox(hGMaxHWnd, "Failed to open RenderWare",BSPExporterTitle, MB_OK );
        delete(m_rwlib);    
        return (TRUE);
    }

    //zero export counts and write version information to warning dialogue
    m_WarningsList.zeroMetrics();
    m_WarningsList.writeVersion();

    /* create an RtImport structure */
    m_importWorld = RtWorldImportCreate();
    if (!m_importWorld)
    {
        MessageBox(hGMaxHWnd, "Unable to allocate RtWorldImport",BSPExporterTitle, MB_OK );
        delete(m_rwlib);    
        return (TRUE);
    }

    /* Calculate the number of output verts and polygons and request space
       in the import world */
    {
        RwInt32 importpolycount = 0, importvertcount = 0;

        AddPolyVertCountInNodeAndChildren(rootNode, &importpolycount, &importvertcount);
        if(!RtWorldImportAddNumVertices(m_importWorld, importvertcount))
        {
            MessageBox(hGMaxHWnd, "Unable to allocate vertices in RtWorldImport",
                BSPExporterTitle, MB_OK );
            RtWorldImportDestroy(m_importWorld);
            delete(m_rwlib);        
            return (FALSE);
        }
        if(!RtWorldImportAddNumTriangles(m_importWorld, importpolycount))
        {
            MessageBox(hGMaxHWnd, "Unable to allocate faces in RtWorldImport",
                BSPExporterTitle, MB_OK );
            RtWorldImportDestroy(m_importWorld);
            delete(m_rwlib);    
            return (FALSE);
        }
    }

    m_numAddedVertices = 0;
    m_numAddedTriangles = 0;

    if (AddNodeToWorld(rootNode))
    {
        m_importWorld->numVertices = m_numAddedVertices;
        m_importWorld->numPolygons = m_numAddedTriangles;
        status = SaveTheWorld(filename);
    }

    char temp[256];
    sprintf(temp,"Total number of vertices = %i",m_numAddedVertices);
    m_WarningsList.add(CWarning(wtInformational,"",temp));
    sprintf(temp,"Total number of triangles = %i",m_numAddedTriangles);
    m_WarningsList.add(CWarning(wtInformational,"",temp));

    //export finished, list warnings & information
    if (m_generateReportFile) {
        string ReportFilename;
        FILE *fd;

        ReportFilename=ChangeFileExtension(filename,ReportFileExtension);
                
        fd=fopen(ReportFilename.c_str(),"w");
        if (fd) {
            //write out the export time
            CTime theTime = CTime::GetCurrentTime();
            CString s = theTime.Format( "%A, %B %d, %Y %H:%M" );
            LPSTR p=s.GetBuffer(256);
            fprintf(fd,"Exported on %s\n",p);
            s.ReleaseBuffer();

            //write out status of export buttons on UI
            fprintf(fd,"Scale Factor=%f\n",dlg.m_nScaleFactor);
            fprintf(fd,"Prelight World=%i\n",dlg.m_bPreLightWorld);
            fprintf(fd,"World sector max polys=%i\n",dlg.m_nWorldSectorMaxPolys);
            fprintf(fd,"World sector size=%f\n",dlg.m_nWorldSectorSize);
            fprintf(fd,"Max plane check=%i\n",dlg.m_maxPlaneCheck);
            fprintf(fd,"Weld Threshold=%f\n",dlg.m_weldThreshold);	
	        fprintf(fd,"Tri strip mesh=%i\n",dlg.m_triStripMesh);
	        fprintf(fd,"Max sector overlap=%f\n",dlg.m_maxSectorOverlap);
	        fprintf(fd,"No alpha in overlap=%i\n",dlg.m_noAlphaInOverlap);
	        fprintf(fd,"Limit UVs=%i\n",dlg.m_limitUVs);
	        fprintf(fd,"Limit UV Max=%i\n",dlg.m_limitUVMax);
	        fprintf(fd,"Weld threshold UV=%f\n",dlg.m_weldThresholdUV);
	        fprintf(fd,"Backface normals=%i\n",dlg.m_backfaceNormals);
	        fprintf(fd,"Two sided materials=%i\n",dlg.m_2SidedMaterials);
	        fprintf(fd,"Color vertex prelight=%i\n",dlg.m_colorVertexPrelight);
	        fprintf(fd,"Weld vertices=%i\n",dlg.m_weldVertices);
	        fprintf(fd,"Texture name case=%i\n",dlg.m_texturenameCase);
	        fprintf(fd,"Condition geometry=%i\n",dlg.m_conditionGeometry);
	        fprintf(fd,"Export normals=%i\n",dlg.m_exportNormals);
	        fprintf(fd,"World offset X=%f\n",dlg.m_worldOffsetX);
	        fprintf(fd,"World offset Y=%f\n",dlg.m_worldOffsetY);
	        fprintf(fd,"Worlf offset Z=%f\n",dlg.m_worldOffsetZ);
	        fprintf(fd,"Display report=%i\n",dlg.m_displayReport);
	        fprintf(fd,"Generate report file=%i\n",dlg.m_generateReportFile);
            fprintf(fd,"\n");

            fprintf(fd,"Warning list:\n");
            m_WarningsList.saveToFile(fd);
            fclose(fd);
        }
    }
    if (!suppressPrompts) {
      if (m_displayReport) {
          CWarningsDlg WarningsDlg;
          WarningsDlg.setWarningsList(&m_WarningsList);
          WarningsDlg.DoModal();
      }
    }

    RtWorldImportDestroy(m_importWorld);

    /* Close RenderWare */
    delete(m_rwlib);    

    return (status);
}

void
BSPExport::AddPolyVertCountInNodeAndChildren(INode *node, RwInt32 *polys, RwInt32 *verts)
{
    int childNum;
    int numChildren;
    Object *object;

    /* node info */
    m_currentNode = node;
    m_currentNodeName = node->GetName();
    m_nodeMaterial = node->GetMtl();
    m_nodeLTM = node->GetObjTMAfterWSM(0);
    m_nodeLTMNoTrans = m_nodeLTM;
    m_nodeLTMNoTrans.NoTrans();
    m_nodeLTMNoTrans.NoScale();

    object = node->EvalWorldState(0).obj;
    if (object && object->IsRenderable() && node->Renderable() )
    {
        if (object->CanConvertToType(triObjectClassID))
        {
            TriObject *triObject = (TriObject *)object->ConvertToType(0, triObjectClassID);
            Mesh *mesh = &triObject->mesh;

            m_reMapper.DoRemap(mesh, m_nodeMaterial, m_limitUVs, m_limitUVMax,
                               m_weldThreshold, m_weldThresholdUV, m_2SidedMaterials, 
                               m_backfaceNormals, m_weldVertices, &m_WarningsList);
            *verts += m_reMapper.GetNumVertices();
            *polys += m_reMapper.GetNumFaces();            
            m_reMapper.ResetReMap();
        }
    }

    /* save the children */
    numChildren = node->NumberOfChildren();
    for (childNum = 0; childNum < numChildren; childNum++)
    {
        INode *childNode;
        
        childNode = node->GetChildNode(childNum);
        AddPolyVertCountInNodeAndChildren(childNode, polys, verts);
    }
}

BOOL
BSPExport::AddNodeToWorld(INode *node)
{
    assert(node);

    int childNum;
    int numChildren;
    Object *object;

    /* node info */
    m_currentNode = node;
    m_currentNodeName = node->GetName();
    m_nodeMaterial = node->GetMtl();
    m_nodeLTM = node->GetObjTMAfterWSM(0);
    m_nodeLTMNoTrans = m_nodeLTM;
    m_nodeLTMNoTrans.NoTrans();
    m_nodeLTMNoTrans.NoScale();

    object = node->EvalWorldState(0).obj;
    if (object && object->IsRenderable() && node->Renderable())
    {
        if (object->CanConvertToType(triObjectClassID))
        {
            TriObject *triObject = (TriObject *)object->ConvertToType(0, triObjectClassID);
            Mesh *mesh = &triObject->mesh;

            AddMeshToWorld(mesh);
        }
    }

    /* save the children */
    numChildren = node->NumberOfChildren();
    for (childNum = 0; childNum < numChildren; childNum++)
    {
        INode *childNode;
        
        childNode = node->GetChildNode(childNum);
        if (!AddNodeToWorld(childNode))
        {
            return (FALSE);
        }
    }

    return (TRUE);
}

BOOL
BSPExport::AddMeshToWorld(Mesh *mesh)
{
    int OldVertices=m_numAddedVertices;
    int OldTriangles=m_numAddedTriangles;
    assert(mesh);

    m_currentMeshVertexIndOffset = m_numAddedVertices;    
    
    m_reMapper.DoRemap(mesh, m_nodeMaterial, m_limitUVs, m_limitUVMax,
                       m_weldThreshold, m_weldThresholdUV, m_2SidedMaterials, 
                       m_backfaceNormals, m_weldVertices, &m_WarningsList);

    if (!SetVertices(mesh))
    {
        return (FALSE);
    }

    if (!SetFaces(mesh))
    {
        return (FALSE);
    }

    m_reMapper.ResetReMap();

    //Write a message stating what was exported
    char temp[256];
    sprintf(temp,"Vertices=%i, Triangles=%i",m_numAddedVertices-OldVertices,
        m_numAddedTriangles-OldTriangles);
    m_WarningsList.add(CWarning(wtInformational,m_currentNodeName,temp));

    return (TRUE);
}


BOOL
BSPExport::SetVertices(Mesh *mesh)
{
    assert(mesh);

    RtWorldImportVertex *vertices;
    int vertexNum;    
    int numVertices = m_reMapper.GetNumVertices();    
    float selfIllum = getMeshSelfIllumination(mesh, m_nodeMaterial);
        
    mesh->buildNormals();

    vertices = RtWorldImportGetVertices(m_importWorld);    
    vertices += m_numAddedVertices;                    

    for (vertexNum = 0; vertexNum < numVertices; vertexNum++)
    {
        ReMapperVertex *remapVertex = m_reMapper.GetVertex(vertexNum);
        Point3 vertex = mesh->getVert(remapVertex->vertexIndex) * m_nodeLTM;

        vertex.x += m_globalWorldOffset.x;
        vertex.y += m_globalWorldOffset.y;
        vertex.z += m_globalWorldOffset.z;

        vertices->OC.x = (RwReal)(vertex.x * m_nScaleFactor);
        vertices->OC.y = (RwReal)(vertex.z * m_nScaleFactor);
        vertices->OC.z = -(RwReal)(vertex.y * m_nScaleFactor);

        if (m_colorVertexPrelight && remapVertex->colorIndex != -1)
        {
            vertices->preLitCol.red = (RwUInt8)(mesh->vertCol[remapVertex->colorIndex].x * 255.0f);
            vertices->preLitCol.green = (RwUInt8)(mesh->vertCol[remapVertex->colorIndex].y * 255.0f);
            vertices->preLitCol.blue = (RwUInt8)(mesh->vertCol[remapVertex->colorIndex].z * 255.0f);
            vertices->preLitCol.alpha = 255;
        }
        else
        {
            vertices->preLitCol.red = 0;
            vertices->preLitCol.green = 0;
            vertices->preLitCol.blue = 0;
            vertices->preLitCol.alpha = 255;
        }
                                           
        vertices->texCoords.u = (RwReal)(remapVertex->u);
        vertices->texCoords.v = (RwReal)(remapVertex->v);

        {
            /* Get the normal */
            Point3 normal;

            if (remapVertex->smGroup != 0)            
            {                            
                RVertex *rvert = mesh->getRVertPtr(remapVertex->vertexIndex);             
                if ((rvert->rFlags & NORCT_MASK) > 1)                            
                {                                
                    unsigned int i;                 
                    for (i = 0; i < (rvert->rFlags & NORCT_MASK); i++)                                
                    {                                    
                        if (rvert->ern[i].getSmGroup() == remapVertex->smGroup)                     
                        {                                        
                            rvert->ern[i].normalize();                         
                            normal = rvert->ern[i].getNormal();                                        
                            break;                                    
                        }                    
                    }                     
                }                     
                else                 
                {                    
                    rvert->rn.normalize();                    
                    normal = rvert->rn.getNormal();                            
                }                        
            }                 
            else             
            {                
                normal = mesh->getFaceNormal(remapVertex->faceIndex);             
            }

            
            if (remapVertex->backface)           
            {                            
                normal *= -1.0f;             
            }
            normal = normal * m_nodeLTMNoTrans;
            vertices->normal.x = normal.x;               
            vertices->normal.y = normal.z;
            vertices->normal.z = -normal.y;
        }

        vertices++;
    }

    m_numAddedVertices += numVertices;

    return (TRUE);
}

BOOL
BSPExport::SetFaces(Mesh *mesh)
{
    assert(mesh);

    int numFaces;
    int faceNum;
    RtWorldImportTriangle *faces;    
    BOOL mirrored = m_nodeLTM.Parity();
    int lastMatID = -1;
    RwInt32 lastMatIndex = -1;
    
    faces = RtWorldImportGetTriangles(m_importWorld);
    faces += m_numAddedTriangles;    
        
    numFaces = m_reMapper.GetNumFaces();
    for (faceNum = 0; faceNum < numFaces; faceNum++)
    {        
        ReMapperFaceLoop *remapFace = m_reMapper.GetFace(faceNum);
        
        /* If we've moved to a new material see if it's already in the
           world, if so reuse it (but check it's mask). If not create
           a new one */
        if (lastMatID != mesh->faces[remapFace->faceNum].getMatID())
        {
            matSearch info;
            RpMaterial *material;

            info.material = NULL;
            info.maxMaterial = m_nodeMaterial;
            info.maxMatID = mesh->faces[remapFace->faceNum].getMatID();
            info.nameCase = m_texturenameCase;

            RtWorldImportForAllMaterials(m_importWorld, CompareRpMaterial2MaxMaterial, (void *)&info);

            if (info.material == NULL)
            {
                material = Max2RpMaterial(m_nodeMaterial, info.maxMatID,
                                            m_texturenameCase);
                lastMatIndex = RtWorldImportAddMaterial(m_importWorld, material);
                RpMaterialDestroy(material);
            }
            else
            {
                material = info.material;
                CheckMaterialMask(material, info.maxMaterial, info.maxMatID, m_texturenameCase);
                lastMatIndex = RtWorldImportGetMaterialIndex(m_importWorld, material);
            }
            
            lastMatID = info.maxMatID;                        
        }

        if (mirrored)
        {
            faces->vertIndex[1] = remapFace->index[0] + m_currentMeshVertexIndOffset;
            faces->vertIndex[0] = remapFace->index[1] + m_currentMeshVertexIndOffset;
        }
        else
        {
            faces->vertIndex[0] = remapFace->index[0] + m_currentMeshVertexIndOffset;
            faces->vertIndex[1] = remapFace->index[1] + m_currentMeshVertexIndOffset;
        }
        faces->vertIndex[2] = remapFace->index[2] + m_currentMeshVertexIndOffset;            
        faces->matIndex = lastMatIndex;
        faces++;             
    }

    m_numAddedTriangles += numFaces;

    return (TRUE);
}

DWORD WINAPI NullProgressFunc(LPVOID arg) {
    return(0);
}

RwBool
RtWorldImportCallback(RwInt32 msg, RwReal value)
{
    switch(msg)
    {
        case rtWORLDIMPORTPROGRESSBSPBUILDSTART:
            BSPExportIP->ProgressStart(_T("Building World Sectors"), TRUE, NullProgressFunc, NULL);
            break;
        case rtWORLDIMPORTPROGRESSBSPBUILDUPDATE:
            BSPExportIP->ProgressUpdate((int)value);
            break;
        case rtWORLDIMPORTPROGRESSBSPBUILDEND:
            BSPExportIP->ProgressEnd();
            break;            
        case rtWORLDIMPORTPROGRESSBSPCOMPRESSSTART:
            BSPExportIP->ProgressStart(_T("Compressing World Sectors"), TRUE, NullProgressFunc, NULL);
            break;
        case rtWORLDIMPORTPROGRESSBSPCOMPRESSUPDATE:
            BSPExportIP->ProgressUpdate((int)value);
            break;
        case rtWORLDIMPORTPROGRESSBSPCOMPRESSEND:
            BSPExportIP->ProgressEnd();
            break;
    }

    return true;
}


RwBool
WorldHasColoredMaterials(RpWorld *world)
{
    RwBool colored = FALSE;

    RpWorldForAllMaterials(world, checkMaterialColors, &colored);

    return colored;
}

RpMesh *
CheckMeshForMaterialEffects(RpMesh *mesh, RpMeshHeader *meshHeader, void *pData)
{
    RwBool *effects = (RwBool *)pData;

    checkMaterialEffects(mesh->material, pData);
    
    if (*effects)
    {
        return NULL;
    }

    return mesh;
}

RpWorldSector *
WorldSectorCheckMaterialEffects(RpWorldSector *worldSector, void *pData)
{
    RwBool effects = FALSE;

    RpWorldSectorForAllMeshes(worldSector, CheckMeshForMaterialEffects, &effects);

    if (effects)
    {
        RpMatFXWorldSectorEnableEffects(worldSector);
    }

    return worldSector;
}


BOOL
BSPExport::SaveTheWorld(const char *filename)
{
    RpWorld *world;
    RtWorldImportParameters worldConvert;
    RwStream *stream;

    /* convert nswWorld to a world */
    assert(m_importWorld);

    /* check the world isn't empty */
    if ((!m_importWorld->numVertices) || (!m_importWorld->numPolygons))
    {
        MessageBox(hGMaxHWnd, "Empty RtWorldImport",BSPExporterTitle, MB_OK );
        return (FALSE);
    }
    
    RtWorldImportParametersInitialize(&worldConvert);    
    worldConvert.maxClosestCheck = m_maxPlaneCheck;
    worldConvert.weldThreshold = m_weldThreshold;
    worldConvert.worldSectorMaxSize = m_nWorldSectorSize;
    worldConvert.maxWorldSectorPolygons = m_nWorldSectorMaxPolys;
    worldConvert.maxOverlapPercent = m_maxSectorOverlap / 100.0f;
    worldConvert.noAlphaInOverlap = m_noAlphaInOverlap;
    worldConvert.conditionGeometry = m_conditionGeometry;    
    worldConvert.fixTJunctions = TRUE;
    if (m_conditionGeometry)
    {        
        worldConvert.weldPolygons = TRUE;
        worldConvert.retainCreases = TRUE;
    }
    worldConvert.calcNormals = FALSE;
    if (m_triStripMesh)
    {
        worldConvert.flags |= rpWORLDTRISTRIP;
    }    
    
    RtWorldImportSetProgressCallBack(RtWorldImportCallback);
        
    world = RtWorldImportCreateWorld(m_importWorld, &worldConvert);

    RtWorldImportSetProgressCallBack(NULL);
    if (!world)
    {
        MessageBox(hGMaxHWnd, "Failed to convert RtWorldImport to RpWorld",
            BSPExporterTitle, MB_OK );
        return (FALSE);
    }

    RpCollisionWorldBuildData(world, NULL);

    if (WorldHasColoredMaterials(world))
    {
        RpWorldSetFlags(world, RpWorldGetFlags(world) | rpWORLDMODULATEMATERIALCOLOR);
    }    

    RpWorldForAllWorldSectors(world, WorldSectorCheckMaterialEffects, NULL);

    /* save the world */
    if ((stream = RwStreamOpen(rwSTREAMFILENAME, rwSTREAMWRITE, (void *)filename)) == NULL)
    {
        MessageBox(hGMaxHWnd, "Failed to write out world",BSPExporterTitle, MB_OK );
        return (FALSE);
    }

    if (!m_exportNormals)
    {
        /* we don't want to write out the normals so remove the flag here */
        RpWorldSetFlags(world, RpWorldGetFlags(world) & ~rpWORLDNORMALS);
    }    
    RpWorldStreamWrite(world, stream);
    if (!m_exportNormals)
    {
        /* replace the normals flag here in case destruction is modified to 
           depend upon the flags */
        RpWorldSetFlags(world, RpWorldGetFlags(world) | rpWORLDNORMALS);
    }
    RwStreamClose(stream, NULL);
    RpWorldDestroy(world);

    return (TRUE);
}
